<?php
// app/Http/Controllers/TangkiController.php

namespace App\Http\Controllers;

use App\Models\Tangki;
use App\Models\Desa;
use App\Models\Dusun;
use App\Models\User;
use App\Models\Notifikasi;
use App\Services\AIPredictionService;
use Illuminate\Http\Request;

class TangkiController extends Controller
{
    protected $aiService;
    
    public function __construct(AIPredictionService $aiService)
    {
        $this->middleware('auth');
        $this->aiService = $aiService;
    }
    
    /**
     * Display tangki schedule
     */
    public function index()
    {
        $user = auth()->user();
        
        $tangkiQuery = Tangki::with('desa', 'dusun', 'petugas')
            ->orderBy('prioritas', 'desc')
            ->orderBy('tanggal_pengiriman', 'asc');
        
        // Filter by role
        if ($user->role === 'petugas_tangki') {
            $tangkiQuery->where('petugas_id', $user->id)
                        ->orWhere('petugas_id', null);
        } elseif ($user->role === 'pemdes' && $user->desa_id) {
            $tangkiQuery->where('desa_id', $user->desa_id);
        }
        
        $tangki = $tangkiQuery->paginate(20);
        
        // Get optimization schedule
        $aiSchedule = [];
        if ($user->isAdmin() || $user->isPemdes()) {
            $aiSchedule = $this->aiService->optimizeTangkiSchedule();
        }
        
        // Data for form
        $desa = Desa::all();
        $petugas = User::where('role', 'petugas_tangki')->where('is_active', true)->get();
        
        return view('dashboard.tangki', compact('tangki', 'aiSchedule', 'desa', 'petugas'));
    }
    
    /**
     * Store new tangki schedule (Admin/Pemdes only)
     */
    public function store(Request $request)
    {
        if (!auth()->user()->isAdmin() && !auth()->user()->isPemdes()) {
            return redirect()->back()->with('error', 'Akses ditolak.');
        }
        
        $validated = $request->validate([
            'desa_id' => 'required|exists:desa,id',
            'dusun_id' => 'nullable|exists:dusun,id',
            'petugas_id' => 'nullable|exists:users,id',
            'tanggal_pengiriman' => 'required|date',
            'waktu_pengiriman' => 'nullable|date_format:H:i',
            'volume_air' => 'required|integer|min:100',
            'prioritas' => 'required|integer|min:1|max:10',
            'catatan' => 'nullable|string',
        ]);
        
        $validated['status'] = 'dijadwalkan';
        
        $tangki = Tangki::create($validated);
        
        // Notify petugas
        if ($validated['petugas_id']) {
            Notifikasi::create([
                'user_id' => $validated['petugas_id'],
                'judul' => 'Jadwal Pengiriman Tangki Baru',
                'pesan' => "Anda dijadwalkan mengirim tangki air ke {$tangki->desa->nama_desa} pada {$tangki->tanggal_pengiriman}",
                'tipe' => 'penting',
            ]);
        }
        
        // Notify desa
        Notifikasi::create([
            'desa_id' => $validated['desa_id'],
            'judul' => 'Tangki Air Dijadwalkan',
            'pesan' => "Tangki air {$validated['volume_air']} liter akan dikirim pada {$validated['tanggal_pengiriman']}",
            'tipe' => 'info',
        ]);
        
        return redirect()->back()->with('success', 'Jadwal tangki berhasil dibuat!');
    }
    
    /**
     * Update tangki status
     */
    public function updateStatus(Request $request, $id)
    {
        $validated = $request->validate([
            'status' => 'required|in:dijadwalkan,dalam_perjalanan,selesai,dibatalkan',
        ]);
        
        $tangki = Tangki::findOrFail($id);
        
        // Check authorization
        $user = auth()->user();
        if (!$user->isAdmin() && 
            !$user->isPemdes() && 
            $tangki->petugas_id !== $user->id) {
            return redirect()->back()->with('error', 'Akses ditolak.');
        }
        
        $tangki->update($validated);
        
        // Notify desa
        Notifikasi::create([
            'desa_id' => $tangki->desa_id,
            'judul' => 'Status Tangki Diperbarui',
            'pesan' => "Pengiriman tangki ke {$tangki->desa->nama_desa} status: {$validated['status']}",
            'tipe' => $validated['status'] === 'selesai' ? 'info' : 'penting',
        ]);
        
        return redirect()->back()->with('success', 'Status tangki berhasil diperbarui!');
    }
    
    /**
     * Assign petugas to tangki
     */
    public function assignPetugas(Request $request, $id)
    {
        if (!auth()->user()->isAdmin() && !auth()->user()->isPemdes()) {
            return redirect()->back()->with('error', 'Akses ditolak.');
        }
        
        $validated = $request->validate([
            'petugas_id' => 'required|exists:users,id',
        ]);
        
        $tangki = Tangki::findOrFail($id);
        $tangki->update($validated);
        
        // Notify petugas
        Notifikasi::create([
            'user_id' => $validated['petugas_id'],
            'judul' => 'Tugas Pengiriman Tangki',
            'pesan' => "Anda ditugaskan mengirim tangki ke {$tangki->desa->nama_desa} pada {$tangki->tanggal_pengiriman}",
            'tipe' => 'penting',
        ]);
        
        return redirect()->back()->with('success', 'Petugas berhasil ditugaskan!');
    }
    
    /**
     * Delete tangki schedule
     */
    public function destroy($id)
    {
        if (!auth()->user()->isAdmin()) {
            return redirect()->back()->with('error', 'Hanya admin yang bisa menghapus jadwal.');
        }
        
        $tangki = Tangki::findOrFail($id);
        $tangki->delete();
        
        return redirect()->back()->with('success', 'Jadwal tangki berhasil dihapus!');
    }
    
    /**
     * Generate AI-optimized schedule
     */
    public function generateAISchedule()
    {
        if (!auth()->user()->isAdmin()) {
            return redirect()->back()->with('error', 'Akses ditolak.');
        }
        
        $schedule = $this->aiService->optimizeTangkiSchedule();
        
        $created = 0;
        foreach ($schedule as $item) {
            Tangki::create([
                'desa_id' => $item['desa_id'],
                'tanggal_pengiriman' => $item['tanggal_rekomendasi'],
                'volume_air' => $item['volume_recommended'],
                'prioritas' => $item['prioritas'],
                'status' => 'dijadwalkan',
                'catatan' => "AI Generated - Risiko Score: {$item['risiko_score']}",
            ]);
            $created++;
        }
        
        return redirect()->back()->with('success', 
            "Berhasil generate {$created} jadwal tangki berdasarkan AI!");
    }
}
